/* -*- Mode: C -*-
 * $Id: gnome-canvas-handle.c,v 1.3 1999/10/30 17:59:49 gregm Exp $
 * Copyright 1998 Free Software Foundation
 *
 * Developed by Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "gnome-canvas-handle.h"

#include <libgnomeui/gnome-canvas-util.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <math.h>
#include <gtk/gtksignal.h>

#include "debug.h"

/****************************************************************************
 * Forward references
 **/
static void    gnome_canvas_handle_class_init (GnomeCanvasHandleClass *class);
static void    gnome_canvas_handle_init       (GnomeCanvasHandle      *handle);
static void    gnome_canvas_handle_destroy    (GtkObject              *object);
static void    gnome_canvas_handle_set_arg    (GtkObject              *object,
					       GtkArg                 *arg,
					       guint                  arg_id);
static void    gnome_canvas_handle_get_arg    (GtkObject              *object,
					       GtkArg                 *arg,
					       guint                  arg_id);
static void    gnome_canvas_handle_realize    (GnomeCanvasItem        *item);
static void    gnome_canvas_handle_unrealize  (GnomeCanvasItem        *item);
static void    gnome_canvas_handle_translate  (GnomeCanvasItem        *item, 
					       double                 dx, 
					       double                 dy);
static gint    gnome_canvas_handle_event      (GnomeCanvasItem        *item, 
					       GdkEvent               *event);
static double  gnome_canvas_handle_point      (GnomeCanvasItem        *item, 
					       double                 x, 
					       double                 y, 
					       int                    cx, 
					       int                    cy, 
					       GnomeCanvasItem        **act_item);
static void    gnome_canvas_handle_draw       (GnomeCanvasItem      *item, 
					       GdkDrawable          *drawable,
					       int                  x, 
					       int                  y, 
					       int                  width, 
					       int                  height);
static void    gnome_canvas_handle_render     (GnomeCanvasItem      *item,
					       GnomeCanvasBuf       *buf);
static void    gnome_canvas_handle_update     (GnomeCanvasItem      *item,
					       double               *affine,
					       ArtSVP               *clip,
					       int                  flags);
static void    gnome_canvas_handle_bounds     (GnomeCanvasItem      *item, 
					       double               *x1, 
					       double               *yy1, 
					       double               *x2, 
					       double               *y2);
static void   get_bounds                      (GnomeCanvasHandle    *handle,
					       double               *px1,
					       double               *py1,
					       double               *px2,
					       double               *py2);
static void   recalc_coords                   (GnomeCanvasHandle    *handle, 
					       double               *x1, 
					       double               *yy1, 
					       double               *x2, 
					       double               *y2);
static void   gtk_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE (GtkObject *object,
					       GtkSignalFunc        func,
					       gpointer             func_data,
					       GtkArg               *args);
/****************************************************************************
 * Local definitions
 **/
/** Arguments */
enum {
  ARG_0,
  ARG_PRELIGHT,
  ARG_ANCHOR,
  ARG_SENSITIVE,
  ARG_X, 
  ARG_Y
};
/** Signals */
enum {
  MOVED,
  LAST_SIGNAL
};
static gint handle_signals[LAST_SIGNAL] = { 0 };
#define HALF_A_HANDLE 3
/** Parent class */
static GnomeCanvasItemClass *parent_class;
/**
 * gnome_canvas_handle_get_type:
 *
 * Registers the &GnomeCanvasHandle class if necessary, and returns the unique
 * type ID associated to it.
 *
 * Return value: The unique type ID of the &GnomeCanvasHandle class.
 **/
GtkType
gnome_canvas_handle_get_type (void)
{
  static GtkType handle_type = 0;
  D_FUNC_START;
  if (!handle_type) 
    {
    static const GtkTypeInfo handle_info = {
      "GnomeCanvasHandle",
      sizeof (GnomeCanvasHandle),
      sizeof (GnomeCanvasHandleClass),
      (GtkClassInitFunc) gnome_canvas_handle_class_init,
      (GtkObjectInitFunc) gnome_canvas_handle_init,
      NULL, /* reserved_1 */
      NULL, /* reserved_2 */
      (GtkClassInitFunc) NULL
    };

    handle_type = gtk_type_unique (gnome_canvas_item_get_type (), &handle_info);
  }
  D_FUNC_END;
  return handle_type;
}
/****************************************************************************
 * Class initialization.
 **/
static void
gnome_canvas_handle_class_init (GnomeCanvasHandleClass *klass)
{
  GtkObjectClass *object_class;
  GnomeCanvasItemClass *item_class;
  D_FUNC_START;
  object_class = (GtkObjectClass *) klass;
  item_class = (GnomeCanvasItemClass *) klass;

  parent_class = gtk_type_class (gnome_canvas_item_get_type ());

  gtk_object_add_arg_type ("GnomeCanvasHandle::prelight", GTK_TYPE_BOOL, 
			   GTK_ARG_READWRITE, ARG_PRELIGHT);
  gtk_object_add_arg_type ("GnomeCanvasHandle::sensitive", GTK_TYPE_BOOL, 
			   GTK_ARG_READWRITE, ARG_SENSITIVE);
  gtk_object_add_arg_type ("GnomeCanvasHandle::x", GTK_TYPE_DOUBLE, 
			   GTK_ARG_READWRITE, ARG_X);
  gtk_object_add_arg_type ("GnomeCanvasHandle::y", GTK_TYPE_DOUBLE, 
			   GTK_ARG_READWRITE, ARG_Y);
  gtk_object_add_arg_type ("GnomeCanvasHandle::anchor", GTK_TYPE_ANCHOR_TYPE, 
			   GTK_ARG_READWRITE, ARG_ANCHOR);

  handle_signals[MOVED] =
    gtk_signal_new ("moved",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCanvasHandleClass, moved),
		    gtk_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE,
		    GTK_TYPE_NONE, 4, 
		    GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE,
		    GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gtk_object_class_add_signals (object_class, handle_signals, 
				LAST_SIGNAL);

  klass->moved = NULL;

  item_class->realize     = gnome_canvas_handle_realize;
  item_class->unrealize   = gnome_canvas_handle_unrealize;
  item_class->update      = gnome_canvas_handle_update;
  item_class->event       = gnome_canvas_handle_event;
  item_class->point       = gnome_canvas_handle_point;
  item_class->translate   = gnome_canvas_handle_translate;
  item_class->draw        = gnome_canvas_handle_draw;
  item_class->render      = gnome_canvas_handle_render;
  item_class->bounds      = gnome_canvas_handle_bounds;

  object_class->destroy   = gnome_canvas_handle_destroy;
  object_class->set_arg   = gnome_canvas_handle_set_arg;
  object_class->get_arg   = gnome_canvas_handle_get_arg;
  D_FUNC_END;
}
/****************************************************************************
 * Object initialization.
 **/
static void
gnome_canvas_handle_init (GnomeCanvasHandle *handle)
{
  D_FUNC_START;
  handle->gc        = NULL;
  handle->svp       = NULL;
  handle->x         = 0.0;
  handle->y         = 0.0;
  handle->anchor    = GTK_ANCHOR_CENTER;
  handle->sensitive = TRUE;
  handle->prelight  = FALSE;
  handle->dragging  = FALSE;
  D_FUNC_END;
}
/****************************************************************************
 * destroy handler.
 **/
static void
gnome_canvas_handle_destroy (GtkObject *object)
{
  GnomeCanvasHandle *handle;
  D_FUNC_START;
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_CANVAS_HANDLE (object));

  handle = GNOME_CANVAS_HANDLE (object);

  if (handle->gc)
    {
      gdk_gc_unref(handle->gc);
      handle->gc = NULL;
    }
  if (handle->svp)
    art_svp_free (handle->svp);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
  D_FUNC_END;
}
/****************************************************************************
 * get_bounds
 **/
static void
get_bounds (GnomeCanvasHandle *handle, double *px1, double *py1, 
	    double *px2, double *py2)
{
  GnomeCanvasItem    *item;
  double             wx, wy;
  double             width;
  double             half;

  item = (GnomeCanvasItem *)handle;
  wx = handle->x;
  wy = handle->y;
  width = 1/item->canvas->pixels_per_unit;
  half  = HALF_A_HANDLE/item->canvas->pixels_per_unit;
  if (width < 0.25)
    width = 0.25;

  gnome_canvas_item_i2w (item, &wx, &wy);
  //  gnome_canvas_w2c (item->canvas, wx + width, wy + width, &wx, &wy);
  switch (handle->anchor)
    {
    case GTK_ANCHOR_NW:
    case GTK_ANCHOR_W:
    case GTK_ANCHOR_SW:
      *px1 = wx - half*2 - width;
      *px2 = wx - width;
      break;
    case GTK_ANCHOR_N:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_S:
      *px1 = wx - half;
      *px2 = wx + half;
      break;
    case GTK_ANCHOR_NE:
    case GTK_ANCHOR_E:
    case GTK_ANCHOR_SE:
      *px1 = wx + width;
      *px2 = wx + half*2 + width;
      break;
    }
  switch (handle->anchor)
    {
    case GTK_ANCHOR_NW:
    case GTK_ANCHOR_N:
    case GTK_ANCHOR_NE:
      *py1 = wy - half*2 - width;
      *py2 = wy - width;
      break;
    case GTK_ANCHOR_W:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_E:
      *py1 = wy - half;
      *py2 = wy + half;
      break;
    case GTK_ANCHOR_SW:
    case GTK_ANCHOR_S:
    case GTK_ANCHOR_SE:
      *py1 = wy + width;
      *py2 = wy + half*2 + width;
      break;
    }
  d_print (DEBUG_DUMP, "wx, wy (%.2f, %.2f / %.2f)\n", wx, wy, width);
}
/****************************************************************************
 * bounds handler
 **/
static void
gnome_canvas_handle_bounds (GnomeCanvasItem *item, double *x1, double *yy1, 
			    double *x2, double *y2)
{
  GnomeCanvasHandle* handle = GNOME_CANVAS_HANDLE (item);
  D_FUNC_START;
  recalc_coords (handle, x1, yy1, x2, y2);
  D_FUNC_END;
}
/****************************************************************************
 * GtkArg Set handler.
 **/
static void
gnome_canvas_handle_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  GnomeCanvasHandle *handle;
  D_FUNC_START;
  handle = GNOME_CANVAS_HANDLE (object);

  switch (arg_id) 
    {
    case ARG_PRELIGHT:
      handle->prelight = GTK_VALUE_BOOL(*arg);
      break;
    case ARG_SENSITIVE:
      handle->sensitive = GTK_VALUE_BOOL(*arg);
      break;
    case ARG_X:
      handle->x = GTK_VALUE_DOUBLE (*arg);
      break;
    case ARG_Y:
      handle->y = GTK_VALUE_DOUBLE (*arg);
      break;
    case ARG_ANCHOR:
      handle->anchor = GTK_VALUE_ENUM (*arg);
      break;
    default:
      g_warning("GnomeCanvasHandle got an unknown arg type.");
      break;
    }
  gnome_canvas_item_request_update ((GnomeCanvasItem *)handle);
  D_FUNC_END;
}
/****************************************************************************
 * GtkArg get handler.
 **/
static void
gnome_canvas_handle_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  GnomeCanvasHandle *handle;
  D_FUNC_START;
  handle = GNOME_CANVAS_HANDLE (object);
  
  switch (arg_id) 
    {
    case ARG_PRELIGHT:
      GTK_VALUE_BOOL (*arg) = handle->prelight;
      break;
    case ARG_SENSITIVE:
      GTK_VALUE_BOOL (*arg) = handle->sensitive;
      break;
    case ARG_X:
      GTK_VALUE_DOUBLE (*arg) = handle->x;
      break;
    case ARG_Y:
      GTK_VALUE_DOUBLE (*arg) = handle->y;
      break;
    case ARG_ANCHOR:
      GTK_VALUE_ENUM (*arg) = handle->anchor;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
  D_FUNC_END;
}
/****************************************************************************
 * realize handler.
 **/
static void
gnome_canvas_handle_realize (GnomeCanvasItem *item)
{
  GdkColor color;
  GnomeCanvasHandle* handle = GNOME_CANVAS_HANDLE(item);
  D_FUNC_START;
  if (GNOME_CANVAS_ITEM_CLASS(parent_class)->realize)
    (* GNOME_CANVAS_ITEM_CLASS(parent_class)->realize) (item);
  
  if (!item->canvas->aa)
    {
      handle->gc = gdk_gc_new (item->canvas->layout.bin_window);
  
      gdk_gc_set_line_attributes (handle->gc, 1, 
				  GDK_LINE_SOLID, 
				  GDK_CAP_PROJECTING, 
				  GDK_JOIN_MITER);
      if (!gdk_color_white(item->canvas->cc->colormap, &color)) 
      g_warning("Couldn't get white color for GnomeCanvasHandle!");
  
      gdk_gc_set_foreground(handle->gc, &color);
      gdk_gc_set_function(handle->gc, GDK_XOR);
    }

  D_FUNC_END;
}
/****************************************************************************
 * unrealize handler.
 **/
static void 
gnome_canvas_handle_unrealize (GnomeCanvasItem *item)
{
  GnomeCanvasHandle* handle = GNOME_CANVAS_HANDLE(item);
  D_FUNC_START;
  if (!item->canvas->aa)
    {
      gdk_gc_unref(handle->gc);
      handle->gc = NULL;
    }
  
  if (GNOME_CANVAS_ITEM_CLASS(parent_class)->unrealize)
    (* GNOME_CANVAS_ITEM_CLASS(parent_class)->unrealize) (item);
  D_FUNC_END;
}
/****************************************************************************
 * update handler.
 **/
static void
gnome_canvas_handle_update (GnomeCanvasItem *item, double *affine, 
			    ArtSVP *clip, int flags)
{
  GnomeCanvasHandle  *handle;
  ArtVpath  vpath[11];
  ArtVpath  *vpath2;
  double  x0, yy0, x1, yy1;
  int     i;
  double  width;

  D_FUNC_START;
  if (parent_class->update)
    (* parent_class->update) (item, affine, clip, flags);
  handle = (GnomeCanvasHandle *)item;
  recalc_coords (handle, &x0, &yy0, &x1, &yy1);
  if (item->canvas->aa)
    {
      gnome_canvas_item_reset_bounds (item);
       i = 0;
       vpath[i].code = ART_MOVETO;
       vpath[i].x = x0;
       vpath[i].y = yy0;
       i++;
       vpath[i].code = ART_LINETO;
       vpath[i].x = x0;
       vpath[i].y = yy1;
       i++;
       vpath[i].code = ART_LINETO;
       vpath[i].x = x1;
       vpath[i].y = yy1;
       i++;
       vpath[i].code = ART_LINETO;
       vpath[i].x = x1;
       vpath[i].y = yy0;
       i++;
       vpath[i].code = ART_LINETO;
       vpath[i].x = x0;
       vpath[i].y = yy0;
       i++;
       
       
       width = 1/item->canvas->pixels_per_unit; /* 1 pixel */
       
       if (width < 0.25) /* copied from rect-ellipse, not understood though. */
	 width = 0.25;
       
       if (!handle->prelight && 
	   x1 - width > x0 + width &&
	   yy1 - width > yy0 + width) 
	 {      
	   vpath[i].code = ART_MOVETO;
	   vpath[i].x = x0 + width;
	   vpath[i].y = yy0 + width;
	   i++;
	   vpath[i].code = ART_LINETO;
	   vpath[i].x = x1 - width;
	   vpath[i].y = yy0 + width;
	   i++;
	   vpath[i].code = ART_LINETO;
	   vpath[i].x = x1 - width;
	   vpath[i].y = yy1 - width;
	   i++;
	   vpath[i].code = ART_LINETO;
	   vpath[i].x = x0 + width;
	   vpath[i].y = yy1 - width;
	   i++;
	   vpath[i].code = ART_LINETO;
	   vpath[i].x = x0 + width;
	   vpath[i].y = yy0 + width;
	   i++;
	 }
       vpath[i].code = ART_END;
       vpath[i].x = 0;
       vpath[i].y = 0;
       
       vpath2 = art_vpath_affine_transform (vpath, affine);
       
       gnome_canvas_item_update_svp_clip (item, &handle->svp, 
					  art_svp_from_vpath (vpath2), 
					  clip);
       art_free (vpath2);
    }
  else
    {
      get_bounds ( (GnomeCanvasHandle *) item, &x0, &yy0, &x1, &yy1);
      d_print (DEBUG_DUMP, "(%.2f, %.2f) - (%.2f, %.2f)\n",
	       x0, yy0, x1, yy1);
      gnome_canvas_update_bbox (item, (int)x0, (int)yy0, (int)x1, (int)yy1);
    }
  D_FUNC_END;
}
/****************************************************************************
 * Gdk draw handler.
 **/
static void   
gnome_canvas_handle_draw  (GnomeCanvasItem *item, GdkDrawable *drawable, 
			   int x, int y, int width, int height)
{
  GnomeCanvasHandle* handle;
  double dx, dy;
  double x1, yy1, x2, y2;
  int cx1, cy1, cx2, cy2;
  gboolean with_prelight;
  D_FUNC_START;
  handle = GNOME_CANVAS_HANDLE (item);

  g_return_if_fail(handle->gc != NULL);
  
  dx = 0.0;
  dy = 0.0;

  d_print (DEBUG_DUMP, "#1 - (%d, %d) - (%d, %d)\n", x, y, width, height);
  gnome_canvas_item_i2w (item, &dx, &dy);
  get_bounds (handle, &x1, &yy1, &x2, &y2);

  gnome_canvas_w2c (item->canvas, dx + x1, dy + yy1, &cx1, &cy1);
  gnome_canvas_w2c (item->canvas, dx + x2, dy + y2, &cx2, &cy2);
  d_print (DEBUG_DUMP, "w2c - (%d, %d) - (%d, %d)\n", cx1, cy1, cx2, cy2);
  with_prelight = handle->prelight && handle->sensitive;
  d_print (DEBUG_DUMP, "#2 - (%d, %d) - (%d, %d)\n", 
	   cx1 - x, cy1 - y, 
	   cx2 - cx1 + (with_prelight ? 1 : 0),
	   cy2 - cy1 + (with_prelight ? 1 : 0));
	   
  gdk_draw_rectangle (drawable,
		      handle->gc,
		      with_prelight,
		      cx1 - x,
		      cy1 - y,
		      /* Filled rectangle covers one fewer so we have to add one */
		      cx2 - cx1 + (with_prelight ? 1 : 0),
		      cy2 - cy1 + (with_prelight ? 1 : 0));  
  D_FUNC_END;
}
/****************************************************************************
 * antialias render handler.
 **/
static void
gnome_canvas_handle_render (GnomeCanvasItem *item, GnomeCanvasBuf *buf)
{

  GnomeCanvasHandle *handle;
  D_FUNC_START;
  handle = (GnomeCanvasHandle *)item;
  if (handle->svp)
    {
      guint color = 0xff;
      gnome_canvas_render_svp (buf, handle->svp, color);
    }
  D_FUNC_END;
}

static void
recalc_coords (GnomeCanvasHandle *handle, 
	       double* x1, double* yy1, double* x2, double* y2)
{
  GnomeCanvasItem *item;
  
  double wx, wy;
  double half;
  double onepixel;
  D_FUNC_START;
  g_return_if_fail(x1  != NULL);
  g_return_if_fail(yy1 != NULL);
  g_return_if_fail(x2  != NULL);
  g_return_if_fail(y2  != NULL);
  
  item = GNOME_CANVAS_ITEM (handle);
  
  wx = handle->x;
  wy = handle->y;
  
  onepixel = 1/item->canvas->pixels_per_unit;

  half = HALF_A_HANDLE/item->canvas->pixels_per_unit;
  
  /* Anchor handle */
  
  switch (handle->anchor) 
    {
    case GTK_ANCHOR_NW:
    case GTK_ANCHOR_W:
    case GTK_ANCHOR_SW:
      /* onepixel prevents overlap with dragged item */
      *x1 = wx - half*2 - onepixel;
      *x2 = wx - onepixel;
      break;
      
    case GTK_ANCHOR_N:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_S:
      *x1 = wx - half;
      *x2 = wx + half;
      break;
      
    case GTK_ANCHOR_NE:
    case GTK_ANCHOR_E:
    case GTK_ANCHOR_SE:
      *x1 = wx + onepixel;
      *x2 = wx + half*2 + onepixel;
      break;
    }
  
  switch (handle->anchor) 
    {
    case GTK_ANCHOR_NW:
    case GTK_ANCHOR_N:
    case GTK_ANCHOR_NE:
      *yy1 = wy - half*2 - onepixel;
      *y2 = wy - onepixel;
      break;
      
    case GTK_ANCHOR_W:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_E:
      *yy1 = wy - half;
      *y2 = wy + half;
      break;
      
    case GTK_ANCHOR_SW:
    case GTK_ANCHOR_S:
    case GTK_ANCHOR_SE:
      *yy1 = wy + onepixel;
      *y2 = wy + half*2 + onepixel;
      break;
    }
  D_FUNC_END;
}

static void 
gnome_canvas_handle_translate   (GnomeCanvasItem *item, double dx, double dy)
{
  GnomeCanvasHandle* handle = GNOME_CANVAS_HANDLE(item);
  D_FUNC_START;
  handle->x += dx;
  handle->y += dy;
  gnome_canvas_item_request_update (item);
  D_FUNC_END;
}

static void
switch_cursors_and_grab(GnomeCanvasHandle* handle, guint32 grab_time)
{
  D_FUNC_START;
  if (!handle->dragging) 
    {
      GdkCursor* ptr;
      
      ptr = NULL;
      switch (handle->anchor) 
	{
	case GTK_ANCHOR_NW:
	  ptr = gdk_cursor_new (GDK_TOP_LEFT_CORNER);
	  break;
	case GTK_ANCHOR_W:
	  ptr = gdk_cursor_new (GDK_LEFT_SIDE);
	  break;
	case GTK_ANCHOR_SW:
	  ptr = gdk_cursor_new (GDK_BOTTOM_LEFT_CORNER);
	  break;
	case GTK_ANCHOR_N:
	  ptr = gdk_cursor_new (GDK_TOP_SIDE);
	  break;
	case GTK_ANCHOR_CENTER:
	  ptr = gdk_cursor_new (GDK_CROSSHAIR);
	  break;
	case GTK_ANCHOR_S:
	  ptr = gdk_cursor_new (GDK_BOTTOM_SIDE);
	  break;      
	case GTK_ANCHOR_NE:
	  ptr = gdk_cursor_new (GDK_TOP_RIGHT_CORNER);
	  break;
	case GTK_ANCHOR_E:
	  ptr = gdk_cursor_new (GDK_RIGHT_SIDE);
	  break;
	case GTK_ANCHOR_SE:
	  ptr = gdk_cursor_new (GDK_BOTTOM_RIGHT_CORNER);
	  break;
	  
	default:
	  g_warning("Bad anchor");
	  break;
	}
      
      gnome_canvas_item_grab (GNOME_CANVAS_ITEM(handle),
			      GDK_POINTER_MOTION_MASK | 
			      GDK_BUTTON_RELEASE_MASK | 
			      GDK_BUTTON_PRESS_MASK,
			      ptr,
			      grab_time);
      
      gdk_cursor_destroy (ptr);
      
      handle->dragging = TRUE;
    }
  D_FUNC_END;
}

static void
ungrab(GnomeCanvasHandle* handle, guint32 ungrab_time)
{
  D_FUNC_START;
  if (handle->dragging) 
    {
      gnome_canvas_item_ungrab(GNOME_CANVAS_ITEM(handle), ungrab_time);
      handle->dragging = FALSE;
    }
  D_FUNC_END;
}
/****************************************************************************
 * Event handler.
 **/
static gint 
gnome_canvas_handle_event(GnomeCanvasItem* item, GdkEvent* event)
{
  GnomeCanvasHandle* handle = GNOME_CANVAS_HANDLE(item);
  gboolean need_redraw = FALSE;
  D_FUNC_START;
  switch (event->type)
    {
      
    case GDK_ENTER_NOTIFY:
      handle->prelight = TRUE;
      need_redraw = TRUE;
      break;
      
    case GDK_LEAVE_NOTIFY:
      handle->prelight = FALSE;
      need_redraw = TRUE;
      break;
      
    case GDK_BUTTON_PRESS:
      if (!handle->sensitive) 
	return FALSE;
      switch (event->button.button) 
	{
	case 1:
	  switch_cursors_and_grab(handle, event->button.time);
	  break;
	  
	default:
	  return FALSE;
	  break;
	}
      break; /* Button press */
      
    case GDK_BUTTON_RELEASE:
      switch (event->button.button) 
	{
	case 1:
	  if (!handle->dragging) 
	    return FALSE;
	  else 
	    ungrab(handle, event->button.time);
	  break;
	  
	default:
	  return FALSE;
	  break;
	}
      break; /* Button release */
      
    case GDK_MOTION_NOTIFY:
      if (handle->dragging) 
	{
	  double newx, newy, dx, dy;
	  
	  /* Now move the handle, only on the dimension(s) allowed
	     by the anchor type, snapping as needed. */
	  newx = event->motion.x;
	  newy = event->motion.y;
	  dx = 0.0;
	  dy = 0.0;
	  gnome_canvas_item_w2i(item, &newx, &newy);
	  
	  if (handle->snap_func) 
	    {
	      (*(handle->snap_func))(&newx, &newy, handle->snap_func_data);
	    }
	  
	  switch (handle->anchor) 
	    {
	    case GTK_ANCHOR_NW:
	    case GTK_ANCHOR_SW:
	    case GTK_ANCHOR_W:
	    case GTK_ANCHOR_NE:
	    case GTK_ANCHOR_SE:
	    case GTK_ANCHOR_E:
	    case GTK_ANCHOR_CENTER:
	      dx = newx - handle->x;
	      handle->x = newx;
	      break;
	      
	    case GTK_ANCHOR_N:
	    case GTK_ANCHOR_S:
	      /* Can't move in X direction */
	      break;
	      
	    default:
	      g_warning("Bad anchor");
	      break;
	    }
	  
	  switch (handle->anchor) 
	    {
	    case GTK_ANCHOR_NW:
	    case GTK_ANCHOR_NE:
	    case GTK_ANCHOR_N:
	    case GTK_ANCHOR_SW:
	    case GTK_ANCHOR_SE:
	    case GTK_ANCHOR_S:
	    case GTK_ANCHOR_CENTER:
	      dy = newy - handle->y;
	      handle->y = newy;
	      break;
	      
	    case GTK_ANCHOR_W:
	    case GTK_ANCHOR_E:
	      /* Can't move in Y direction */
	      break;
	      
	    default:
	      g_warning("Bad anchor");
	      break;
	    }
	  gtk_signal_emit(GTK_OBJECT(handle), handle_signals[MOVED], 
			  handle->x, handle->y, dx, dy);
	  need_redraw = TRUE;
	}
      else 
	return FALSE;
      break; /* Motion notify */
      
    default:
      return FALSE;
    }
  
  /* Draw new location or redraw with/without fill */
  if (need_redraw)
    gnome_canvas_item_request_update ( (GnomeCanvasItem *)handle);
  D_FUNC_END;
  return TRUE;
}

void    
gnome_canvas_handle_set_snap_func(GnomeCanvasHandle* handle, 
				  GnomeCanvasSnapFunc func, 
				  gpointer data)
{
  D_FUNC_START;
  handle->snap_func = func;
  handle->snap_func_data = data;
  D_FUNC_END;
}

static double
gnome_canvas_handle_point (GnomeCanvasItem *item, double x, double y, 
			   int cx, int cy, GnomeCanvasItem **actual_item)
{
  GnomeCanvasHandle *handle;
  double x1, yy1, x2, y2;
  double dx, dy;
  D_FUNC_START;
  handle = GNOME_CANVAS_HANDLE (item);

  *actual_item = item;

  recalc_coords (handle, &x1, &yy1, &x2, &y2);

  /* On top of the handle */

  if ((x >= x1) && (y >= yy1) && (x <= x2) && (y <= y2)) 
    {
      return 0.0;
    }
  
  /* Point is outside */

  if (x < x1)
    dx = x1 - x;
  else if (x > x2)
    dx = x - x2;
  else
    dx = 0.0;
  
  if (y < yy1)
    dy = yy1 - y;
  else if (y > y2)
    dy = y - y2;
  else
    dy = 0.0;
  D_FUNC_END;
  return sqrt (dx * dx + dy * dy);
}


typedef void (*GtkSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE) (GtkObject *object,
							     gdouble arg1,
							     gdouble arg2,
							     gdouble arg3,
							     gdouble arg4,
							     gpointer data);

static void gtk_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE (GtkObject * object,
							   GtkSignalFunc func,
							   gpointer func_data,
							   GtkArg * args)
{

  GtkSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE rfunc;
  D_FUNC_START;
  rfunc = (GtkSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE) func;
  (*rfunc) (object,
	    GTK_VALUE_DOUBLE (args[0]),
	    GTK_VALUE_DOUBLE (args[1]),
	    GTK_VALUE_DOUBLE (args[2]),
	    GTK_VALUE_DOUBLE (args[3]),
	    func_data);
  D_FUNC_END;
}
/* EOF */


