/*
**  poprox_proxy.c
**
**  Copyright 1996,1997 - Board of Trustees - University of Illinois - Urbana
**  Jason Wessel - jwessel@uiuc.edu Computing & Communications Services Office
**  All rights reserved. 
**
**  Copyright 1998 - Board of Trustees - University of Illinois at Chicago
**  John Schulien - jms@uic.edu Academic Computing and Communications Center
**  All rights reserved
**   
**  This program may be freely modified and distributed provided that this
**  copyright information is retained.
*/

#include "poprox.h"


/*
**  A subroutine to format a read packet and send it to syslog.  This 
**  subroutine is invoked when DEBUG_PACKETS is active.  
**  A packet may contain multiple lines seperated by '\r\n', so we need to
**  scan the buffer and break apart the lines to write them to syslog.
*/

static void
logpacket(hdr1, hdr2, buf, size)
char *hdr1;				/* Initial header in syslog */
char *hdr2;				/* Continuation header in syslog */
char *buf;				/* Pointer to buffer */
int size;
{
    char *c = buf;                      /* Working pointer */
    char *h = hdr1;			/* Initial header string */
    buf[size] = '\0';                   /* Delimit with zero */
    while (size > 0) {				/* Loop until buffer empty */ 
	while (size && *c && *c != '\n') {      /* While !eol and !'\n' */ 
	    c++; 				/* Bump to next character */
	    size--;				/* Adjust byte count */
        }
        *c = '\0';				/* Delimit with '\0' */
        syslog(LOG_INFO, "%s %s", h, buf);	/* Write to log */
        c++; 					/* Advance past '\0' */
        buf = c; 				/* Start of next line */
        size--;					/* Adjust size */
	h = hdr2;				/* Continuation header */
   }
}

/*
**  Here is the subroutine that does the proxy operation
**
**  Return codes are as follows:
**
**  PROXY_RC_OK		-  The proxy operation completed successfully
**  PROXY_RC_TIMEOUT    -  The connection timed out
**  PROXY_RC_SOCKET	-  Unable to create a server socket
**  PROXY_RC_SOCKOPT	-  setsockopt() failed on server socket
**  PROXY_RC_BIND	-  Unable to bind the server socket
**  PROXY_RC_RES_NAME	-  Unable to resolve the server name
**  PROXY_RC_RES_ADDR   -  Unable to resolve the server address
**  PROXY_RC_RES_SS     -  Error from select() on server socket
**
*/

int 
poprox_proxy(pophostname, pophostport, command, timeout)
char 		* pophostname;	/* Host to connect this user to */
int		  pophostport;	/* Port to connect to on server */
char		* command;	/* Command that the user entered */
struct timeval  * timeout;	/* Seconds before we time out */
{

    SOCKADDR_IN sin;		/* Socket address structure */
    SOCKET	csoc;		/* POP Client Socket */
    SOCKET	ssoc;		/* POP Server Socket */
    struct hostent *hp;		/* Used to locate POP server */

    fd_set 	readfds;	/* File descriptor set for reading */
    int 	nfds;		/* # of file descriptors in a set */

    int 	bytes;		/* Bytes read into the buffer */
    int 	rc;		/* Return code from various functions */
    int 	one = 1;	/* Used for calls to setsockopt() */

    char 	*buf;		/* The transmit/receive buffer */	
    int		buflen;		/* Length of the transmit/receive buffer */

/*
**  Allocate storage for our buffer
*/

    buf = (char *)malloc(BUFSIZE + 1);
    buflen = BUFSIZE;

/*
**  Prepare the POP client socket, which is always stdout.
**  Note that we do NOT want to perform error checking on the
**  setsockopt() for the client socket, because if poproxd is
**  invoked from the command line (for testing purposes), it will
**  fail with ENOTSOCK (socket operation on non-socket.) This is ok.
*/

    csoc = STDOUT_FILENO;		/* Client socket already exists */
    setsockopt(csoc, 			/* Set the socket options */
	       IPPROTO_TCP, 		/* Disable the Nagle algorithm */
	       TCP_NODELAY,		/* to prevent send coalescing */
	       (void *) &one, 		/* The value to set is 1 */
	       sizeof(one));		/* Length of the value field */

/*
**  Create a socket to connect to the pop server, and set the 
**  TCP_NODELAY option to avoid send coalescing.
*/

    ssoc = socket(AF_INET, SOCK_STREAM, 0); /* Create the Server socket */
    if (ssoc == INVALID_SOCKET) {           /* If the operation failed ... */
	syslog(LOG_INFO, "Error %d attempting to create remote socket, errno");
	return PROXY_RC_SOCKET;
    }

    if (setsockopt(ssoc,		/* Set the socket options */
	       IPPROTO_TCP,             /* Disable the Nagle algorithm */
	       TCP_NODELAY,             /* to prevent send coalescing */
	       (void *) &one,           /* The value to set is 1 */
	       sizeof(one))) {          /* Length of the value field */
	syslog(LOG_INFO, "Error %d from setsockopt on remote socket", errno);
	return PROXY_RC_SOCKOPT;
    }

    bzero((char *)&sin, sizeof(sin));	/* Zero the address structure */
    sin.sin_family = AF_INET;		/* This is an internet address */
    sin.sin_addr.s_addr = INADDR_ANY;	/* No specific address needed */
    
    if (bind(ssoc, &sin, sizeof(sin))) {    /* Attempt to bind the socket */
	syslog(LOG_INFO, "Error %d attempting to bind remote socket", errno);
	return PROXY_RC_BIND;
    }

/*
**  Now obtain the internet address of the target POP server and
**  attempt to connect to it.
*/

    hp = gethostbyname(pophostname);	/* Try to resolve the host name */ 
    if (hp == NULL) {			/* If we failed ... */
	syslog(LOG_INFO, "Unable to resolve remote server name");
	return PROXY_RC_RES_NAME;
    }

    if (sin.sin_addr.s_addr == INADDR_NONE) {   /* If we couldn't resolve ... */
	syslog(LOG_INFO, "Unable to resolve remote server address");
	return PROXY_RC_RES_ADDR;
    }

    bzero((char *)&sin, sizeof(sin));	/* Zero the address structure */
    sin.sin_family = AF_INET;		/* Set the address family */
    sin.sin_port = htons(pophostport);	/* Set the port to bind to */
    memcpy(&sin.sin_addr.s_addr, 	/* Obtain the host address */
	   hp->h_addr_list[0],		/* from the hostent structure */
	   hp->h_length);		/* Host address is h_length bytes */

    rc = connect(ssoc, 				/* Connect to the pop server */
		 (struct sockaddr *)&sin,	/* using structure from above */
		 sizeof(sin));			/* Size of the structure */
    if (rc) {					/* If we failed ... */
        syslog(LOG_INFO, "Error %d attempting to connect to remote server",
	    errno);
    return PROXY_RC_CONNECT;
    }

/*
**  We are now connected and ready.  Issue a read to absorb the herald
**  message from the server, then send the 'user' command using the
**  username we picked up earlier.
*/

    poprox_fgets(buf, buflen,	    	    /* Address and length of buffer */
		 ssoc, timeout);	    /* Server socket / timeout period */
    write(ssoc,command, strlen(command));   /* Send command to server */
    if (debug & DEBUG_PACKETS) {            /* If packet debugging */
	logpacket("P>S", "P..", buf, strlen(buf)); /* Log the initial command */
    }

/*
**  Now all we need to do is proxy the rest of the traffic between the
**  real server and the client.  We have sent the 'user' command, so the
**  server will reply with the 'password' prompt.  We pass this along to
**  the client, and from here on out we don't need to know the contents 
**  of the client/server traffic.  As a matter of fact, we can't even
**  safely report errors to the client because we don't know if we're in
**  the middle of a transaction or not.  However, we can and do report 
**  errors to syslog.
*/

    nfds = FD_SETSIZE;				    /* Length of readfds */
    while (1) {					    /* Do forever ... */
	FD_ZERO(&readfds);			    /* Start with a null set */
	FD_SET(ssoc,&readfds);			    /* Add server socket */
	FD_SET(csoc,&readfds);			    /* Add client socket */
        rc = select(nfds, &readfds, 0, 0, timeout); /* Wait for some data */
	if (rc == 0) {                  	    /* If we timed out ... */
	    syslog(LOG_INFO, "Terminating due to select timeout condition");
	    poprox_cleanup(0);          	    /* Clean up and exit */
	}
	if (rc < 0) {                   	    /* If we had an error */
	    syslog(LOG_INFO, "Terminating due to select error %d", errno);
	    poprox_cleanup(0);          	    /* Clean up and exit */
	}
	if (FD_ISSET(ssoc,&readfds)) {		    /* If we have server data */
	    bytes = read(ssoc,buf,buflen);	    /* Read from server */
	    if (bytes == 0) {			    /* End of file? */
		syslog(LOG_INFO, "End of file received from server");
		poprox_cleanup(0);		    /* Yes - Quit! */
	    }
	    if (bytes < 0) {			    /* Error on read? */
		syslog(LOG_INFO, 
		       "Terminating due to server read error %d", errno);
		poprox_cleanup(0);		    /* Yes - Quit! */
	    }
	    rc = write(csoc,buf,bytes);	    	    /* Write data to client */
	    if (bytes != rc) {			    /* If an error occurred */
		syslog(LOG_INFO, 
		       "Terminating due to client write error %d", errno);
		poprox_cleanup(0);		    /* Clean up and exit */
	    }
            if (debug & DEBUG_PACKETS)              /* If packet debugging */
		logpacket("S>C", "S..", buf, bytes); /* Log the packet */
	}
	if (FD_ISSET(csoc,&readfds)) {		    /* If we have client data */
	    bytes = read(csoc,buf,buflen);    	    /* Read from server */
	    if (bytes == 0) {			    /* End of file? */
		syslog(LOG_INFO, "End of file received from client");
		poprox_cleanup(0);		    /* Yes - Quit! */
	    }
	    if (bytes < 1) {			    /* Error on read? */
		syslog(LOG_INFO, 
		       "Terminating due to client read error %d", errno);
		poprox_cleanup(0);		    /* Yes - Quit! */
	    }     
	    rc = write(ssoc,buf,bytes);	    	    /* Write data to server */
	    if (bytes != rc) {			    /* If an error occurred */
		syslog(LOG_INFO, 
		       "Terminating due to server write error %d", errno);
		poprox_cleanup(0);		    /* Clean up and exit */
	    }
	    if (debug & DEBUG_PACKETS)              /* If packet debugging */
		logpacket("C>S", "C..", buf, bytes); /* Log the packet */
	}
    }
}


