/*
**  poprox_imap.c
**
**  Copyright 1996,1997 - Board of Trustees - University of Illinois - Urbana
**  Jason Wessel - jwessel@uiuc.edu Computing & Communications Services Office
**  All rights reserved. 
**
**  Copyright 1998 - Board of Trustees - University of Illinois at Chicago
**  John Schulien - jms@uic.edu Academic Computing and Communications Center
**  All rights reserved
**   
**  This program may be freely modified and distributed provided that this
**  copyright information is retained.
*/


#include "poprox.h"

/*
**  This file encapsulates all of the IMAP-specific code in poprox.
**  It contains two subroutines:  poprox_user_imap, and poprox_msg_imap.
**  poprox_user_imap is used to obtain the username and first command
**  from the client, and poprox_msg_imap is used to send an error message
**  to the client and terminate.
*/

/*
**  The following string must match what is returned by the CAPABILITY
**  command on the actual IMAP servers.  Hard coding it is a kludge,
**  because we can't really get the capabilities of the server until
**  we connect to it, and we don't know what our server will be
**  until we get the LOGIN command.  The problem is that eudora asks
**  for the capabilities BEFORE it submits the login.  A better 
**  solution might be to specify a "default" imap server that we can
**  contact to obtain the capability string. 
*/

#define IMAP_CAPABILITY_STRING \
	"* CAPABILITY IMAP4 IMAP4REV1 NAMESPACE"\
	" IDLE SCAN SORT AUTH=LOGIN THREAD=ORDEREDSUBJECT"

/*
**  The following static variable holds the current command tag.
**  the command tag is generated by the client, and may be needed
**  later to print an appropriate error message.
*/

char *tag = NULL;

/*
**  The following subroutine may be called to send a fatal error 
**  message to the IMAP client.  errno should be valid at this point.
*/

void
poprox_msg_imap(msg)
int	msg;			/* Message number */
{
    switch(msg) {

    case PROXY_RC_SOCKET:
	poprox_msg("%s BAD Error %d attempting to create remote socket",
	    tag, errno);
	break;
    case PROXY_RC_SOCKOPT:
	poprox_msg("%s BAD Error %d from setsockopt on remote socket", 
	    tag, errno);
	break;
    case PROXY_RC_BIND:
        poprox_msg("%s BAD Error %d attempting to bind remote socket",
	    tag, errno);
	break;
    case PROXY_RC_RES_NAME:
        poprox_msg("%s BAD Unable to resolve remote server name",
	    tag);
	break;
    case PROXY_RC_RES_ADDR:
	poprox_msg("%s BAD Unable to resolve remote server address",
	    tag);
	break;
    case PROXY_RC_CONNECT:
	poprox_msg("%s BAD Error %d attempting to connect to remote server",
	    tag, errno);
	break;
    case PROXY_RC_DBOPEN:
	poprox_msg("%s BAD Error opening mailbox database",
	    tag);
	break;
    case PROXY_RC_DBFIND:
        poprox_msg("%s BAD Cannot locate your maildrop",
	    tag);
	break;
    case PROXY_RC_DBDATA:
        poprox_msg("%s BAD Your maildrop database entry is invalid",
	    tag);
	break;
    }
    poprox_cleanup(0);		/* Clean up and exit */
}

/*
**  The following routine begins the proxy process for an IMAP session.
**  First, we send a herald to the IMAP client.  The IMAP client will
**  reply with the first command.  The only three valid commands at this
**  point are 'login', 'logout', and 'capability'
**
**  The capability string is sent before the login string, which causes
**  a problem.  The only solution I can think of right now is to
**  make the capability string a configurable part of the proxy server.
**  For now, though, it must be hard coded.  We don't know the 
**  capabilities of the pop server on the remote system because we
**  don't even know what the remote system is.  Sigh.
**
**  Once we have received a login command, we return pointers both to
**  the entered username, which will be used to determine the target
**  server, and to the entire command, which will be forwarded to the
**  server untouched.
*/

int
poprox_user_imap(proxyhostname, timeout, command, username) 
char 		*  proxyhostname;	/* Name of the proxy host */
struct timeval	*  timeout;		/* Seconds before timeout */
char 		** command;		/* Place to put command */
char 		** username;		/* Place to put username */
{
    char line[MAXLINELEN + 1];		/* Read buffer */
    char *c;				/* Pointer to parsed command */
    char *u;				/* Pointer to parsed username */
    char *x;				/* Used to parse next token */

    poprox_msg("* OK %s IMAP4rev1 v12.249 server  proxy ready", proxyhostname);

/*
**  Main loop begins here
*/

    while (1) {				/* Main command loop begins here */

/*
**  Read in the next line from the client and save it.
*/

	if (poprox_fgets(line,		/* Read a line from the POP client */
			MAXLINELEN,	/* maximum line length */
			STDIN_FILENO,	/* Input file descriptor */
			timeout) 	/* Timeout period */
	    == NULL) {			/* If end of file received ... */
	    syslog(LOG_INFO, "Ending session due to end of file condition");
	    return(1);			
	} 
	*command = (char *)malloc(strlen(line) + 1);  /* Copy entire line */
	strcpy(*command, line);			      /* for our caller */

/*
**  The first token should be the command tag.
*/

	if (tag) {				/* If an old tag is present */
	    free(tag);				/* then free it now */
	    tag = NULL;				/* Clear out the pointer */
	}
	x = strtok(line, " \t\n\r");		/* Isolate the command tag */
	if (!x) {				/* No tag given? */
	    poprox_msg("* BAD Null command");
	    free(*command);			/* Release command storage */
	    *command = NULL;			/* Null out the pointer */
	    continue;				/* Missing identifier */
	}
	tag = (char *)malloc(strlen(x) + 1);	/* Copy the command tag */
	strcpy(tag, x);				/* into free storage */

/*
**  The next token should be the command
*/

	c = strtok(NULL, " \t\n\r");		/* Find the command name */
	if (!c) {		               	/* Missing command */
	    poprox_msg("%s BAD Missing command", tag);
	    free(*command);
	    *command = NULL;			
	    continue;
	}
	strupper(c);				/* Uppercase the command */

/*
**  Process the CAPABILITY command if present
*/
	if (!strcmp(c, "CAPABILITY")) {		/* Look for 'capability' */
	    poprox_msg(IMAP_CAPABILITY_STRING);
	    poprox_msg("%s OK CAPABILITY completed.", tag);
	    free(*command);
	    *command = NULL;
	    continue;
	}

/*
**  Process the LOGOUT command if present
*/

	if (!strcmp(c, "LOGOUT")) {     	/* Look for 'logout' */
	    poprox_msg("* BYE %s imap4proxy v2.0 server terminating connection",
		proxyhostname);
	    poprox_msg("%s OK LOGOUT completed", tag);
            return(1);
	}

/*
**  Process the LOGIN command if present
*/

        if (!strcmp(c, "LOGIN")) {  		/* Look for 'login' */
	    u = strtok(NULL, " \t\n\r");	/* Parse the username */
	    if (!u) {				/* Username missing? */
		poprox_msg("%s BAD Missing required argument to LOGIN", tag);
		free(*command);	
		*command = NULL;		
		continue;
	    }
	    x = strtok(NULL, " \t\n\r");	/* Look for password */
	    if (!x) {				/* Password missing? */
		poprox_msg("%s BAD Missing required argument to LOGIN", tag);
		free(*command);
		*command = NULL;
		continue;
	    }
	    x = strtok(NULL, " \t\n\r");	/* PW must be last token */
	    if (x) {				/* More tokens? */
		poprox_msg("%s BAD Argument given to LOGIN when none expected",
		    tag);
		free(*command);			/* Free the command */
		*command = NULL;
		continue;
	    }
	    *username = (char *)malloc(strlen(u) + 1);  /* Copy the username */
	    strcpy(*username, u);			/* for our caller */
	    return 0;					/* Return to caller */
	}

/*
**  No other commands are valid at this point
*/

	poprox_msg("%s BAD Command unrecognized/login please: %s", tag, c);
	free(*command);  
	*command = NULL;
    }				/* loop back to while(1) */
}


